##########################################################################
#
#  Estimating hourly time-series temperature and thermal attributes at
#  a spot-measured site, using hourly time-series temperature data at 
#  another site (e.g. hydrological gauging station).
#
#  This code requires to run the R script "Fourier_1.R" beforehand
#  as this script uses some variables which are generated by the script. 
#
#  Input: an hourly recorded temperature data (text file)
#           1st column for "Time" and 2nd column for "Temperature"
#           "example_data2.txt" is analyzed.
#
# Output: figure for summary and decomposed temperature as csv
#
#                    2016.07    Masahiro Ryo
#                         masahiroryo@gmail.com
#
#
##########################################################################

###############################################################
###     PREPARATION PART
###############################################################
setwd("C:/Users/Public/Documents/")

### to read a spot-measured data and couple with the time-series data
observed <- read.table("example_data2.txt",sep="\t", head=T)
observed[,1] <- as.POSIXct(strptime(observed[,1],format="%Y.%m.%d %H:%M"))
observed <- cbind(observed, data$Temperature[is.element(Time, observed[,1])], ext.factor[is.element(Time, observed[,1])])
colnames(observed) <- c("Time", "Obs", "ts", "noise")  


###################################################################
# To estimate parameters: b1, b2, b3, b4, b5
# see eq.7 in the paper
###################################################################
### To estimate parameters b1 and b5
b1 <- lm(observed$Obs~observed$ts)$"coefficients"[2]  # slope of linear regression
b5 <- lm(observed$Obs~observed$ts)$"coefficients"[1]  # intercept of linear regression

### To estimate parameters b2 and b3 by least squares method
min_mse <- 1.0e+5; b2 <- 0.0; b3 <- 0.0

for(b2.tmp in seq(0.01, 1.0, by = 0.01)){
  for(b3.tmp in seq(0.01, 1.0, by = 0.01)){
    FFT_corrected <- FFT_Temperature_selected
    ### average temperature is adjusted by the slope of linear regression
    FFT_corrected[1] <- b1*FFT_Temperature[1]

    ### seasonal
    FFT_corrected[FFT_seasonal] <- complex(re=b2.tmp*Re(FFT_corrected[FFT_seasonal]),
                                           im=b2.tmp*Im(FFT_corrected[FFT_seasonal]))
    ### diel
    FFT_corrected[FFT_diel] <- complex(re=b3.tmp*Re(FFT_corrected[FFT_diel]),
                                       im=b3.tmp*Im(FFT_corrected[FFT_diel]))

    Temperature_corrected <- Re(fft(FFT_corrected, inverse=TRUE))  # transforming from frequency to time
    Temperature_corrected <- data.frame(data$Time,Temperature_corrected)
    Z_dt  <- observed$Obs - Temperature_corrected[is.element(Time, observed[,1]),2]
    tmp_mse <- sum(lm(observed$noise~Z_dt)$residuals^2)
    if(tmp_mse<min_mse){min_mse <- tmp_mse; b2 <- b2.tmp; b3 <- b3.tmp}
  }  
}

### update FFT_corrected with fitted parameters
FFT_corrected <- FFT_Temperature_selected
FFT_corrected[1] <- b1*FFT_Temperature[1]
FFT_corrected[FFT_seasonal] <- complex(re=b2*Re(FFT_corrected[FFT_seasonal]),
                                       im=b2*Im(FFT_corrected[FFT_seasonal]))
FFT_corrected[FFT_diel] <- complex(re=b3*Re(FFT_corrected[FFT_diel]),
                                   im=b3*Im(FFT_corrected[FFT_diel]))
Temperature_corrected <- Re(fft(FFT_corrected, inverse=TRUE))  # transforming from frequency to time
Temperature_corrected <- data.frame(data$Time,Temperature_corrected)


### To estimate parameters b4 by least squares method
min_mse2 <- 1.0e+5; b4 <- 0.0
for (b4.tmp in seq(0,1, by=0.01)){
  tmp <- Temperature_corrected
  tmp[,2] <- Temperature_corrected[,2] + b4.tmp*ext.factor
  tmp[which(tmp[,2]<0),2] <- tmp[which(tmp[,2]<0),2]  - b5
  tmp[which(tmp[,2]<0),2] <- 0
  Est <- tmp[is.element(Time, observed[,1]),2]
  tmp_mse2 <- sum(lm(observed$Obs~Est)$residuals^2)
  if(tmp_mse2<min_mse2){min_mse2 <- tmp_mse2; b4 <- b4.tmp}
}


### finalize estimation with fitted parameters
tmp[,2] <- Temperature_corrected[,2] + b4*ext.factor + b5 
tmp[which(tmp[,2]<0),2] <- tmp[which(tmp[,2]<0),2]  - b5
tmp[which(tmp[,2]<0),2] <- 0
Timeseries.estimated <- tmp
spot.fitted <- tmp[is.element(Time, observed[,1]),2]

### results
lm.result <- lm(observed$Obs~spot.fitted)
r2 <- summary(lm.result)$"r.squared"

plot(Timeseries.estimated[,2], type="l")
plot(observed$Obs, spot.fitted)  
parameter.estimated <- cbind(round(b1, digits = 2),b2,b3,b4,round(b5, digits = 2))
rownames(parameter.estimated)<- c("")

### exporting results as csv:
write.csv(Timeseries.estimated, "Extrapolation_result.csv")
parameter.estimated
r2 # r-squared between observed and estimated (fitting)
  